﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancing/ElasticLoadBalancingRequest.h>
#include <aws/elasticloadbalancing/ElasticLoadBalancing_EXPORTS.h>
#include <aws/elasticloadbalancing/model/Instance.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancing {
namespace Model {

/**
 * <p>Contains the parameters for
 * DeregisterInstancesFromLoadBalancer.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticloadbalancing-2012-06-01/DeregisterEndPointsInput">AWS
 * API Reference</a></p>
 */
class DeregisterInstancesFromLoadBalancerRequest : public ElasticLoadBalancingRequest {
 public:
  AWS_ELASTICLOADBALANCING_API DeregisterInstancesFromLoadBalancerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeregisterInstancesFromLoadBalancer"; }

  AWS_ELASTICLOADBALANCING_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCING_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the load balancer.</p>
   */
  inline const Aws::String& GetLoadBalancerName() const { return m_loadBalancerName; }
  inline bool LoadBalancerNameHasBeenSet() const { return m_loadBalancerNameHasBeenSet; }
  template <typename LoadBalancerNameT = Aws::String>
  void SetLoadBalancerName(LoadBalancerNameT&& value) {
    m_loadBalancerNameHasBeenSet = true;
    m_loadBalancerName = std::forward<LoadBalancerNameT>(value);
  }
  template <typename LoadBalancerNameT = Aws::String>
  DeregisterInstancesFromLoadBalancerRequest& WithLoadBalancerName(LoadBalancerNameT&& value) {
    SetLoadBalancerName(std::forward<LoadBalancerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the instances.</p>
   */
  inline const Aws::Vector<Instance>& GetInstances() const { return m_instances; }
  inline bool InstancesHasBeenSet() const { return m_instancesHasBeenSet; }
  template <typename InstancesT = Aws::Vector<Instance>>
  void SetInstances(InstancesT&& value) {
    m_instancesHasBeenSet = true;
    m_instances = std::forward<InstancesT>(value);
  }
  template <typename InstancesT = Aws::Vector<Instance>>
  DeregisterInstancesFromLoadBalancerRequest& WithInstances(InstancesT&& value) {
    SetInstances(std::forward<InstancesT>(value));
    return *this;
  }
  template <typename InstancesT = Instance>
  DeregisterInstancesFromLoadBalancerRequest& AddInstances(InstancesT&& value) {
    m_instancesHasBeenSet = true;
    m_instances.emplace_back(std::forward<InstancesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_loadBalancerName;
  bool m_loadBalancerNameHasBeenSet = false;

  Aws::Vector<Instance> m_instances;
  bool m_instancesHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancing
}  // namespace Aws
